/*
 *  $Id: gwyui.c 27960 2025-05-10 13:07:23Z yeti-dn $
 *  Copyright (C) 2003-2025 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"

#include "libgwyddion/gwyddion.h"
#include "libgwyui/gwyui.h"

/**
 * gwy_ui_init:
 *
 * Makes libgwyui types safe for deserialization and performs other initialization.  You have to call this function
 * before using widgets and objects from libgwyui. It calls gwy_init() so you do not have to initialise libgwyddion
 * separately.
 *
 * It is safe to call this function more than once, subsequent calls are generally no-op.
 *
 * It can be used in both GUI and non-GUI programs. The behaviour depends on whether GTK+ has been initialised. If it
 * has and there is a default screen, additional initialisation is done, such as loading Gwyddion icons to the default
 * theme.
 **/
void
gwy_ui_init(void)
{
    static gboolean types_initialized = FALSE, icons_loaded = FALSE;

    if (!types_initialized) {
        gwy_init();

        g_type_class_peek(GWY_TYPE_GRAPH_CURVE_MODEL);
        g_type_class_peek(GWY_TYPE_GRAPH_MODEL);
        g_type_class_peek(GWY_TYPE_GL_LABEL);
        g_type_class_peek(GWY_TYPE_GL_SETUP);

        types_initialized = TRUE;
    }

    if (!icons_loaded) {
        GdkScreen *screen = gdk_screen_get_default();

        if (screen) {
            GtkIconTheme *theme = gtk_icon_theme_get_default();

            gchar *iconpath = gwy_find_self_path("data", "icons", NULL);
            // XXX: They say gtk_icon_theme_add_resource_path() should be used for application-specific icons. But
            // that requires compiling the icons to a resource file, loading it into a GResource and registering said
            // resource. Since we are doing none of that, just prepend our icons to the search path.
            //gtk_icon_theme_add_resource_path(theme, path);
            gtk_icon_theme_prepend_search_path(theme, iconpath);
            g_free(iconpath);

            iconpath = g_build_filename(gwy_get_user_dir(), "icons", NULL);
            if (g_file_test(iconpath, G_FILE_TEST_IS_DIR))
                gtk_icon_theme_add_resource_path(theme, iconpath);
            g_free(iconpath);

            icons_loaded = TRUE;
        }
    }

    /* FIXME GTK3 */
#if 0
    gtk_rc_parse_string
        (/* graph window statusbar */
         "style \"gwyflatstatusbar\" {\n"
         "  GtkStatusbar::shadow_type = 0\n"
         "}\n"
         "widget \"*.gwyflatstatusbar\" style \"gwyflatstatusbar\"\n"
         "\n"
         /* adjust bar internal check button */
         "style \"gwyadjbarcheck\" {\n"
         "  GtkCheckButton::focus_padding = 0\n"
         "  GtkCheckButton::focus_line_width = 0\n"
         "}\n"
         "widget \"*.gwyadjbarcheck\" style \"gwyadjbarcheck\"\n"
         "\n");
#endif
}

/**
 * SECTION: gwyui
 * @title: gwyui
 * @short_description: Library initialisation
 *
 * Gwyddion classes has to be initialized before they can be deserialized. The function gwy_ui_init() performs this
 * initialization.
 **/

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
