/*
 * @(#)Pyraminx2d.c
 *
 * Copyright 1994 - 2024  David A. Bagley, bagleyd AT verizon.net
 *
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the author not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.
 *
 * This program is distributed in the hope that it will be "useful",
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */

/* Methods file for Pyraminx2d */

#include "PyraminxP.h"
#include "Pyraminx2dP.h"

#ifndef WINVER
static Boolean setValuesPuzzle2D(Widget current,
	Widget request, Widget renew);
static void resizePuzzle2D(Pyraminx2DWidget w);
static void initializePuzzle2D(Widget request, Widget renew);
static void exposePuzzle2D(Widget renew,
	XEvent *event, Region region);
static void movePuzzle2DTop(Pyraminx2DWidget w,
	XEvent *event, char **args, int nArgs);
static void movePuzzle2DTr(Pyraminx2DWidget w,
	XEvent *event, char **args, int nArgs);
static void movePuzzle2DLeft(Pyraminx2DWidget w,
	XEvent *event, char **args, int nArgs);
static void movePuzzle2DRight(Pyraminx2DWidget w,
	XEvent *event, char **args, int nArgs);
static void movePuzzle2DBl(Pyraminx2DWidget w,
	XEvent *event, char **args, int nArgs);
static void movePuzzle2DBottom(Pyraminx2DWidget w,
	XEvent *event, char **args, int nArgs);

static char translations2D[] =
"<KeyPress>q: Quit()\n\
 Ctrl<KeyPress>C: Quit()\n\
 <KeyPress>osfCancel: Hide()\n\
 <KeyPress>Escape: Hide()\n\
 <KeyPress>osfEscape: Hide()\n\
 Ctrl<KeyPress>[: Hide()\n\
 <KeyPress>0x1B: Hide()\n\
 <KeyPress>0x2E: Speed()\n\
 <KeyPress>0x3E: Speed()\n\
 <KeyPress>0x3C: Slow()\n\
 <KeyPress>0x2C: Slow()\n\
 Shift<KeyPress>2: Sound()\n\
 <KeyPress>F11: MoveCcw()\n\
 <KeyPress>KP_Divide: MoveCcw()\n\
 <KeyPress>R5: MoveCcw()\n\
 <KeyPress>Up: MoveTop()\n\
 <KeyPress>osfUp: MoveTop()\n\
 <KeyPress>KP_Up: MoveTop()\n\
 <KeyPress>KP_8: MoveTop()\n\
 <KeyPress>R8: MoveTop()\n\
 <KeyPress>Prior: MoveTr()\n\
 <KeyPress>KP_9: MoveTr()\n\
 <KeyPress>R9: MoveTr()\n\
 <KeyPress>Left: MoveLeft()\n\
 <KeyPress>osfLeft: MoveLeft()\n\
 <KeyPress>KP_Left: MoveLeft()\n\
 <KeyPress>KP_4: MoveLeft()\n\
 <KeyPress>R10: MoveLeft()\n\
 <KeyPress>F12: MoveCw()\n\
 <KeyPress>Begin: MoveCw()\n\
 <KeyPress>KP_5: MoveCw()\n\
 <KeyPress>R11: MoveCw()\n\
 <KeyPress>Right: MoveRight()\n\
 <KeyPress>osfRight: MoveRight()\n\
 <KeyPress>KP_Right: MoveRight()\n\
 <KeyPress>KP_6: MoveRight()\n\
 <KeyPress>R12: MoveRight()\n\
 <KeyPress>End: MoveBl()\n\
 <KeyPress>KP_1: MoveBl()\n\
 <KeyPress>R13: MoveBl()\n\
 <KeyPress>Down: MoveBottom()\n\
 <KeyPress>osfDown: MoveBottom()\n\
 <KeyPress>KP_Down: MoveBottom()\n\
 <KeyPress>KP_2: MoveBottom()\n\
 <KeyPress>R14: MoveBottom()\n\
 <Btn1Down>: Select()\n\
 <Btn1Up>: Release()\n\
 <Btn2Down>: PracticeMaybe()\n\
 <Btn2Down>(2+): Practice2()\n\
 <Btn3Down>: RandomizeMaybe()\n\
 <Btn3Down>(2+): Randomize2()\n\
 <Btn4Down>: MoveTop()\n\
 <Btn5Down>: MoveBottom()\n\
 <KeyPress>g: Get()\n\
 <KeyPress>w: Write()\n\
 <KeyPress>u: Undo()\n\
 <KeyPress>r: Redo()\n\
 <KeyPress>c: Clear()\n\
 <KeyPress>z: Randomize()\n\
 <KeyPress>s: Solve()\n\
 <KeyPress>f: Find()\n\
 <KeyPress>p: Practice()\n\
 <KeyPress>i: Increment()\n\
 <KeyPress>d: Decrement()\n\
 <KeyPress>o: Orientize()\n\
 <KeyPress>2: Period2()\n\
 <KeyPress>3: Period3()\n\
 <KeyPress>b: Both()\n\
 <KeyPress>y: Sticky()\n\
 <KeyPress>v: View()\n\
 <EnterWindow>: Enter()\n\
 <LeaveWindow>: Leave()";

static XtActionsRec actionsList2D[] =
{
	{(char *) "Quit", (XtActionProc) quitPuzzle},
	{(char *) "Hide", (XtActionProc) hidePuzzle},
	{(char *) "MoveCcw", (XtActionProc) movePuzzleCcw},
	{(char *) "MoveTop", (XtActionProc) movePuzzle2DTop},
	{(char *) "MoveTr", (XtActionProc) movePuzzle2DTr},
	{(char *) "MoveLeft", (XtActionProc) movePuzzle2DLeft},
	{(char *) "MoveCw", (XtActionProc) movePuzzleCw},
	{(char *) "MoveRight", (XtActionProc) movePuzzle2DRight},
	{(char *) "MoveBl", (XtActionProc) movePuzzle2DBl},
	{(char *) "MoveBottom", (XtActionProc) movePuzzle2DBottom},
	{(char *) "Select", (XtActionProc) selectPuzzle},
	{(char *) "Release", (XtActionProc) releasePuzzle},
	{(char *) "PracticeMaybe", (XtActionProc) practicePuzzleWithQuery},
	{(char *) "Practice2", (XtActionProc) practicePuzzleWithDoubleClick},
	{(char *) "RandomizeMaybe", (XtActionProc) randomizePuzzleWithQuery},
	{(char *) "Randomize2", (XtActionProc) randomizePuzzleWithDoubleClick},
	{(char *) "Get", (XtActionProc) getPuzzle},
	{(char *) "Write", (XtActionProc) writePuzzle},
	{(char *) "Undo", (XtActionProc) undoPuzzle},
	{(char *) "Redo", (XtActionProc) redoPuzzle},
	{(char *) "Clear", (XtActionProc) clearPuzzle},
	{(char *) "Randomize", (XtActionProc) randomizePuzzle},
	{(char *) "Solve", (XtActionProc) solvePuzzle},
	{(char *) "Find", (XtActionProc) findPuzzle},
	{(char *) "Practice", (XtActionProc) practicePuzzle},
	{(char *) "Increment", (XtActionProc) incrementPuzzle},
	{(char *) "Decrement", (XtActionProc) decrementPuzzle},
	{(char *) "Orientize", (XtActionProc) orientizePuzzle},
	{(char *) "Period2", (XtActionProc) period2ModePuzzle},
	{(char *) "Period3", (XtActionProc) period3ModePuzzle},
	{(char *) "Both", (XtActionProc) bothModePuzzle},
	{(char *) "Sticky", (XtActionProc) stickyModePuzzle},
	{(char *) "View", (XtActionProc) viewPuzzle},
	{(char *) "Speed", (XtActionProc) speedUpPuzzle},
	{(char *) "Slow", (XtActionProc) slowDownPuzzle},
	{(char *) "Sound", (XtActionProc) toggleSoundPuzzle},
	{(char *) "Enter", (XtActionProc) enterPuzzle},
	{(char *) "Leave", (XtActionProc) leavePuzzle}
};

static XtResource resources2D[] =
{
	{XtNwidth, XtCWidth, XtRDimension, sizeof (Dimension),
	 XtOffset(PyraminxWidget, core.width),
	 XtRString, (caddr_t) "300"},
	{XtNheight, XtCHeight, XtRDimension, sizeof (Dimension),
	 XtOffset(PyraminxWidget, core.height),
	 XtRString, (caddr_t) "600"},
	{XtNmono, XtCMono, XtRBoolean, sizeof (Boolean),
	 XtOffset(PyraminxWidget, pyraminx.mono),
	 XtRString, (caddr_t) "FALSE"},
	{XtNreverseVideo, XtCReverseVideo, XtRBoolean, sizeof (Boolean),
	 XtOffset(PyraminxWidget, pyraminx.reverse),
	 XtRString, (caddr_t) "FALSE"},
	{XtNforeground, XtCForeground, XtRPixel, sizeof (Pixel),
	 XtOffset(PyraminxWidget, pyraminx.foreground),
	 XtRString, (caddr_t) XtDefaultForeground},
	{XtNbackground, XtCBackground, XtRPixel, sizeof (Pixel),
	 XtOffset(PyraminxWidget, pyraminx.background),
	 XtRString, (caddr_t) "#AEB2C3" /*XtDefaultBackground*/},
	{XtNframeColor, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(PyraminxWidget, pyraminx.frameColor),
	 XtRString, (caddr_t) "Cyan" /*XtDefaultForeground*/},
	/* Beware color values are swapped */
	{XtNfaceColor0, XtCLabel, XtRString, sizeof (String),
	 XtOffset(PyraminxWidget, pyraminx.faceName[0]),
	 XtRString, (caddr_t) "Blue"},
	{XtNfaceColor1, XtCLabel, XtRString, sizeof (String),
	 XtOffset(PyraminxWidget, pyraminx.faceName[1]),
	 XtRString, (caddr_t) "Red"},
	{XtNfaceColor2, XtCLabel, XtRString, sizeof (String),
	 XtOffset(PyraminxWidget, pyraminx.faceName[2]),
	 XtRString, (caddr_t) "Yellow"},
	{XtNfaceColor3, XtCLabel, XtRString, sizeof (String),
	 XtOffset(PyraminxWidget, pyraminx.faceName[3]),
	 XtRString, (caddr_t) "Green"},
	{XtNpieceBorder, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(PyraminxWidget, pyraminx.borderColor),
	 XtRString, (caddr_t) "gray25" /*XtDefaultForeground*/},
	{XtNdelay, XtCDelay, XtRInt, sizeof (int),
	 XtOffset(PyraminxWidget, pyraminx.delay),
	 XtRString, (caddr_t) "10"}, /* DEFAULT_DELAY */
	{XtNsound, XtCSound, XtRBoolean, sizeof (Boolean),
	 XtOffset(PyraminxWidget, pyraminx.sound),
	 XtRString, (caddr_t) "FALSE"},
	{XtNmoveSound, XtCMoveSound, XtRString, sizeof (String),
	 XtOffset(PyraminxWidget, pyraminx.moveSound),
	 XtRString, (caddr_t) MOVESOUND},
	{XtNfont, XtCFont, XtRString, sizeof (String),
	 XtOffset(PyraminxWidget, pyraminx.font),
	 XtRString, (caddr_t) "9x15bold"},
	{XtNsize, XtCSize, XtRInt, sizeof (int),
	 XtOffset(PyraminxWidget, pyraminx.size),
	 XtRString, (caddr_t) "3"}, /*DEFAULT_FACETS */
	{XtNsticky, XtCSticky, XtRBoolean, sizeof (Boolean),
	 XtOffset(PyraminxWidget, pyraminx.sticky),
	 XtRString, (caddr_t) "FALSE"}, /* DEFAULT_STICKY */
	{XtNmode, XtCMode, XtRInt, sizeof (int),
	 XtOffset(PyraminxWidget, pyraminx.mode),
	 XtRString, (caddr_t) "3"}, /*DEFAULT_MODE */
	{XtNorient, XtCOrient, XtRBoolean, sizeof (Boolean),
	 XtOffset(PyraminxWidget, pyraminx.orient),
	 XtRString, (caddr_t) "FALSE"}, /*DEFAULT_ORIENT */
	{XtNpractice, XtCPractice, XtRBoolean, sizeof (Boolean),
	 XtOffset(PyraminxWidget, pyraminx.practice),
	 XtRString, (caddr_t) "TRUE"}, /*DEFAULT_PRACTICE */
	{XtNuserName, XtCUserName, XtRString, sizeof (String),
	 XtOffset(PyraminxWidget, pyraminx.userName),
	 XtRString, (caddr_t) ""},
	{XtNscoreFile, XtCScoreFile, XtRString, sizeof (String),
	 XtOffset(PyraminxWidget, pyraminx.scoreFile),
	 XtRString, (caddr_t) ""},
	{XtNscoreOnly, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(PyraminxWidget, pyraminx.scoreOnly),
	 XtRString, (caddr_t) "FALSE"},
	{XtNversionOnly, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(PyraminxWidget, pyraminx.versionOnly),
	 XtRString, (caddr_t) "FALSE"},
	{XtNmenu, XtCMenu, XtRInt, sizeof (int),
	 XtOffset(PyraminxWidget, pyraminx.menu),
	 XtRString, (caddr_t) "999"}, /* ACTION_IGNORE */
	{XtNstart, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(PyraminxWidget, pyraminx.started),
	 XtRString, (caddr_t) "FALSE"},
	{XtNcheat, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(PyraminxWidget, pyraminx.cheat),
	 XtRString, (caddr_t) "FALSE"},
	{XtNface, XtCFace, XtRInt, sizeof (int),
	 XtOffset(PyraminxWidget, pyraminx.currentFace),
	 XtRString, (caddr_t) "-1"},
	{XtNpos, XtCPos, XtRInt, sizeof (int),
	 XtOffset(PyraminxWidget, pyraminx.currentPosition),
	 XtRString, (caddr_t) "-1"},
	{XtNdirection, XtCDirection, XtRInt, sizeof (int),
	 XtOffset(PyraminxWidget, pyraminx.currentDirection),
	 XtRString, (caddr_t) "-1"},
	{XtNstyle, XtCStyle, XtRInt, sizeof (int),
	 XtOffset(PyraminxWidget, pyraminx.currentStyle),
	 XtRString, (caddr_t) "-1"},
	{XtNcontrol, XtCControl, XtRInt, sizeof (int),
	 XtOffset(PyraminxWidget, pyraminx.currentControl),
	 XtRString, (caddr_t) "0"},
	{XtNfast, XtCFast, XtRInt, sizeof (int),
	 XtOffset(PyraminxWidget, pyraminx.currentFast),
	 XtRString, (caddr_t) "1"},
	{XtNpixmapSize, XtCPixmapSize, XtRInt, sizeof (int),
	 XtOffset(PyraminxWidget, pyraminx.pixmapSize),
	 XtRString, (caddr_t) "64"},
	{XtNselectCallback, XtCCallback, XtRCallback, sizeof (caddr_t),
	 XtOffset(PyraminxWidget, pyraminx.select),
	 XtRCallback, (caddr_t) NULL}
};

Pyraminx2DClassRec pyraminx2dClassRec =
{
	{
		(WidgetClass) & pyraminxClassRec,	/* superclass */
		(char *) "Pyraminx2D",	/* class name */
		sizeof (Pyraminx2DRec),	/* widget size */
		NULL,		/* class initialize */
		NULL,		/* class part initialize */
		FALSE,		/* class inited */
		(XtInitProc) initializePuzzle2D,	/* initialize */
		NULL,		/* initialize hook */
		XtInheritRealize,	/* realize */
		actionsList2D,	/* actions */
		XtNumber(actionsList2D),	/* num actions */
		resources2D,	/* resources */
		XtNumber(resources2D),	/* num resources */
		NULLQUARK,	/* xrm class */
		TRUE,		/* compress motion */
		TRUE,		/* compress exposure */
		TRUE,		/* compress enterleave */
		TRUE,		/* visible interest */
		NULL,		/* destroy */
		(XtWidgetProc) resizePuzzle2D,	/* resize */
		(XtExposeProc) exposePuzzle2D,	/* expose */
		(XtSetValuesFunc) setValuesPuzzle2D,	/* set values */
		NULL,		/* set values hook */
		XtInheritSetValuesAlmost,	/* set values almost */
		NULL,		/* get values hook */
		XtInheritAcceptFocus,		/* accept focus */
		XtVersion,	/* version */
		NULL,		/* callback private */
		translations2D,	/* tm table */
		NULL,		/* query geometry */
		NULL,		/* display accelerator */
		NULL		/* extension */
	},
	{
		0		/* ignore */
	},
	{
		0		/* ignore */
	}
};

WidgetClass pyraminx2dWidgetClass = (WidgetClass) & pyraminx2dClassRec;
#endif

static Point triangleUnit[MAX_SIDES][4] =
{
	{
		{0, 2},
		{1, 0},
		{-1, 1},
		{0, -1}
	},
	{
		{1, 3},
		{-1, 0},
		{1, -1},
		{0, 1}
	}
};
static Point triangleList[MAX_SIDES][4], letterList[MAX_SIDES];
static Point offsetList[MAX_SIDES];

static Pixmap dr = 0; /* dummy for future double buffering */

static void
drawOrientLine2D(Pyraminx2DWidget w, int orient, int dx, int dy,
		int face, int tetra,
		Point *triangleArray, GC borderGC)
{
	int x_1 = 0, x_2 = 0, y_1 = 0, y_2 = 0, offset = 0;
	Point center, corner[3], edge[3];

	corner[0].x = triangleArray[0].x;
	corner[0].y = triangleArray[0].y;
	corner[1].x = corner[0].x + triangleArray[1].x;
	corner[1].y = corner[0].y + triangleArray[1].y;
	corner[2].x = corner[1].x + triangleArray[2].x;
	corner[2].y = corner[1].y + triangleArray[2].y;
	center.x = (corner[0].x + corner[1].x + corner[2].x) / 3;
	center.y = (corner[0].y + corner[1].y + corner[2].y) / 3;
	edge[0].x = (corner[1].x + corner[2].x) / 2;
	edge[0].y = (corner[1].y + corner[2].y) / 2;
	edge[1].x = (corner[2].x + corner[0].x) / 2;
	edge[1].y = (corner[2].y + corner[0].y) / 2;
	edge[2].x = (corner[0].x + corner[1].x) / 2;
	edge[2].y = (corner[0].y + corner[1].y) / 2;
	if ((face & 1) == 1) {
		tetra = (tetra == 1) ? 0 : 1;
		offset = 2;
	}
	orient = orient % 3;
	if (tetra == 1) {
		switch (orient) {
		case 0:
			x_1 = (7 * edge[0].x + center.x) / 8 + offset;
			y_1 = (7 * edge[0].y + center.y) / 8 + offset;
			x_2 = center.x + offset;
			y_2 = center.y + offset;
			break;
		case 1:
			x_1 = (2 * edge[1].x + center.x) / 3;
			y_1 = (2 * edge[1].y + center.y) / 3;
			x_2 = center.x;
			y_2 = center.y;
			break;
		case 2:
			x_1 = (2 * edge[2].x + center.x) / 3;
			y_1 = (2 * edge[2].y + center.y) / 3;
			x_2 = center.x;
			y_2 = center.y;
			break;
		default:
			{
				char *buf;

				intCat(&buf, "drawOrientLine2D(tetra): orient ",
					orient);
				DISPLAY_WARNING(buf);
				free(buf);
			}
		}
	} else {
		switch (orient) {
		case 0:
			x_1 = (corner[0].x + center.x) / 2;
			y_1 = (corner[0].y + center.y) / 2;
			x_2 = center.x;
			y_2 = center.y;
			break;
		case 1:
			x_1 = (corner[1].x + 2 * center.x) / 3;
			y_1 = (corner[1].y + 2 * center.y) / 3;
			x_2 = center.x;
			y_2 = center.y;
			break;
		case 2:
			x_1 = (corner[2].x + 2 * center.x) / 3;
			y_1 = (corner[2].y + 2 * center.y) / 3;
			x_2 = center.x;
			y_2 = center.y;
			break;
		default:
			{
				char *buf;

				intCat(&buf, "drawOrientLine2D(octa): orient ",
					orient);
				DISPLAY_WARNING(buf);
				free(buf);
			}
		}
	}
	DRAWLINE(w, dr, borderGC, x_1, y_1, x_2, y_2);
}

void
drawTriangle2D(Pyraminx2DWidget w, int face, int position, int offset)
{
/*#ifndef FIND*/
	GC faceGC, borderGC;
	int dx, dy, faceOnView, orient, side, view, tetra;
	CRD crd;

	toCRD((PyraminxWidget) w, face, position, &crd);
	tetra = (((position - crd.diagonal * crd.diagonal) % 2) == 0) ? 1 : 0;
	view = (face / MAX_SIDES == UP) ? DOWN : UP;
	side = crd.diagonal - crd.column - crd.row;
	if (!w->pyraminx.vertical && view == DOWN) {
		faceOnView = (side == UP) ? DOWN : UP;
		dx = (w->pyraminx.size - crd.column - 1) *
			(w->pyraminx.facetLength + w->pyraminx.delta);
		dy = (w->pyraminx.size - crd.row - 1) *
			(w->pyraminx.facetLength + w->pyraminx.delta);
		orient = (w->pyraminx.facetLoc[face][position].rotation +
			MAX_ORIENT / 2) % MAX_ORIENT;
		if (2 * w->pyraminx.size - crd.column - crd.row - 2 +
				faceOnView >= w->pyraminx.size) {
			dx += w->pyraminx.sideOffset;
			dy += w->pyraminx.sideOffset;
		}
	} else {
		faceOnView = side;
		dx = crd.column * (w->pyraminx.facetLength + w->pyraminx.delta);
		dy = crd.row * (w->pyraminx.facetLength + w->pyraminx.delta);
		orient = w->pyraminx.facetLoc[face][position].rotation;
		if (crd.column + crd.row + faceOnView >= w->pyraminx.size) {
			dx += w->pyraminx.sideOffset;
			dy += w->pyraminx.sideOffset;
		}
	}
	dx += w->pyraminx.puzzleOffset.x + w->pyraminx.delta;
	dy += w->pyraminx.puzzleOffset.y + w->pyraminx.delta;
	if (view == DOWN) {
		if (w->pyraminx.vertical)
			dy += w->pyraminx.viewLength - w->pyraminx.delta - 1;
		else
			dx += w->pyraminx.viewLength - w->pyraminx.delta - 1;
	}
	triangleList[faceOnView][0].x = offsetList[faceOnView].x + dx;
	triangleList[faceOnView][0].y = offsetList[faceOnView].y + dy;
	if (offset) {
		borderGC = w->pyraminx.faceGC[w->pyraminx.facetLoc[face][position].face];
		if (w->pyraminx.mono) {
			faceGC = w->pyraminx.inverseGC;
		} else {
			faceGC = w->pyraminx.borderGC;
		}
	} else {
		faceGC = w->pyraminx.faceGC[w->pyraminx.facetLoc[face][position].face];
		borderGC = w->pyraminx.borderGC;
	}
	POLYGON(w, dr, faceGC, borderGC, triangleList[faceOnView], 3,
		True, False);
	dx += -faceOnView;
	dy += -faceOnView;
	if (w->pyraminx.mono) {
		int letterX, letterY;
		char buf[2];

		buf[0] =
#ifdef WINVER
			w->pyraminx.faceChar[w->pyraminx.facetLoc
				[face][position].face];
#else
			w->pyraminx.faceName[w->pyraminx.facetLoc
				[face][position].face][0];
#endif
		buf[1] = '\0';
		letterX = dx + letterList[faceOnView].x;
		letterY = dy + letterList[faceOnView].y;
		if (offset) {
			borderGC = w->pyraminx.borderGC;
		} else {
			borderGC = w->pyraminx.inverseGC;
		}
		DRAWTEXT(w, dr, borderGC, letterX, letterY, buf, 1);
	}
	if (w->pyraminx.orient)
		drawOrientLine2D(w, orient, dx, dy, face, tetra,
			triangleList[faceOnView], borderGC);
/*#endif*/
}

void
eraseFrame2D(const Pyraminx2DWidget w)
{
	FILLRECTANGLE(w, dr, w->pyraminx.inverseGC,
		0, 0, w->core.width, w->core.height);
}

void
drawFrame2D(Pyraminx2DWidget w, Boolean focus)
{
	int startx, starty, lengthx, lengthy, longlength;
	GC gc = (focus) ? w->pyraminx.frameGC : w->pyraminx.borderGC;

	startx = 1 + w->pyraminx.puzzleOffset.x;
	starty = 1 + w->pyraminx.puzzleOffset.y;
	lengthx = w->pyraminx.viewLength - w->pyraminx.delta +
		w->pyraminx.puzzleOffset.x;
	lengthy = w->pyraminx.viewLength - w->pyraminx.delta +
		w->pyraminx.puzzleOffset.y;
	DRAWLINE(w, dr, gc, startx, starty, lengthx, starty);
	DRAWLINE(w, dr, gc, startx, starty, startx, lengthy);
	DRAWLINE(w, dr, gc, startx, lengthy, lengthx, starty);
	if (w->pyraminx.vertical) {
		longlength = 2 * w->pyraminx.viewLength - 2 * w->pyraminx.delta - 1 +
			w->pyraminx.puzzleOffset.y;
		DRAWLINE(w, dr, gc,
			startx, lengthy, startx, longlength);
		DRAWLINE(w, dr, gc,
			lengthx, lengthy, lengthx, longlength);
		DRAWLINE(w, dr, gc,
			startx, longlength, lengthx, longlength);
		DRAWLINE(w, dr, gc,
			startx, longlength, lengthx, lengthy);
		DRAWLINE(w, dr, gc,
			lengthx, starty, lengthx, lengthy);
		DRAWLINE(w, dr, w->pyraminx.frameGC,
			0, lengthy, (int) w->core.width, lengthy);
	} else {
		longlength = 2 * w->pyraminx.viewLength - 2 * w->pyraminx.delta - 1 +
			w->pyraminx.puzzleOffset.x;
		DRAWLINE(w, dr, gc,
			lengthx, starty, longlength, starty);
		DRAWLINE(w, dr, gc,
			lengthx, lengthy, longlength, lengthy);
		DRAWLINE(w, dr, gc,
			longlength, starty, longlength, lengthy);
		DRAWLINE(w, dr, gc,
			longlength, starty, lengthx, lengthy);
		DRAWLINE(w, dr, gc,
			startx, lengthy, lengthx, lengthy);
		DRAWLINE(w, dr, w->pyraminx.frameGC,
			lengthx, 0, lengthx, (int) w->core.height);
	}
}

static void
resizePieces(Pyraminx2DWidget w)
{
	int i, j;

	w->pyraminx.facetLength = w->pyraminx.faceLength / w->pyraminx.size -
		w->pyraminx.delta - 1;
	for (i = 0; i <= 3; i++)
		for (j = 0; j < MAX_SIDES; j++) {
			triangleList[j][i].x = triangleUnit[j][i].x *
				w->pyraminx.facetLength;
			triangleList[j][i].y = triangleUnit[j][i].y *
				w->pyraminx.facetLength;
		}
	offsetList[DOWN].x = 0;
	offsetList[UP].x = w->pyraminx.facetLength + 1;
	offsetList[DOWN].y = 0;
	offsetList[UP].y = w->pyraminx.facetLength + 1;
	letterList[DOWN].x = w->pyraminx.facetLength / 4 -
		w->pyraminx.letterOffset.x;
	letterList[UP].x = 3 * w->pyraminx.facetLength / 4 + 3 -
		w->pyraminx.letterOffset.x;
	letterList[DOWN].y = w->pyraminx.facetLength / 4 + 1 +
		w->pyraminx.letterOffset.y;
	letterList[UP].y = 3 * w->pyraminx.facetLength / 4 + 1 +
		w->pyraminx.letterOffset.y;
	w->pyraminx.sideOffset = 3 * w->pyraminx.size / 4;
	w->pyraminx.orientLineLength = w->pyraminx.facetLength / 4;
	w->pyraminx.orientDiagLength = MAX(w->pyraminx.orientLineLength - 3, 0);
}

Boolean
selectPieces2D(Pyraminx2DWidget w, int positionX, int positionY,
		int *face, int *position)
{
	CRD crd;
	int offset, modI, modJ, side, view;
	int x = positionX - w->pyraminx.puzzleOffset.x;
	int y = positionY - w->pyraminx.puzzleOffset.y;

	if (w->pyraminx.vertical && y > w->pyraminx.viewLength - 1) {
		y -= (w->pyraminx.viewLength - 1);
		view = DOWN;
	} else if (!w->pyraminx.vertical && x > w->pyraminx.viewLength - 1) {
		x -= (w->pyraminx.viewLength - 1);
		view = DOWN;
	} else
		view = UP;
	if (x <= 0 || y <= 0 ||
			x >= w->pyraminx.faceLength + w->pyraminx.delta ||
			y >= w->pyraminx.faceLength + w->pyraminx.delta)
		return False;
	else if (x + y > w->pyraminx.faceLength)
		offset = 2 * w->pyraminx.delta + 1;
	else
		offset = w->pyraminx.delta;
	crd.column = (x - offset) /
		(w->pyraminx.facetLength + w->pyraminx.delta);
	crd.row = (y - offset) /
		(w->pyraminx.facetLength + w->pyraminx.delta);
	modI = (x - offset) % (w->pyraminx.facetLength + w->pyraminx.delta);
	modJ = (y - offset) % (w->pyraminx.facetLength + w->pyraminx.delta);
	side = (modI + modJ > w->pyraminx.facetLength + 1) ? UP : DOWN;
	if (!w->pyraminx.vertical && view == DOWN) {
		crd.row = w->pyraminx.size - crd.row - 1;
		crd.column = w->pyraminx.size - crd.column - 1;
		side = (side == UP) ? DOWN : UP;
	}
	crd.diagonal = crd.row + crd.column + side;
	*face = ((view == UP) ? DOWN : UP) * MAX_SIDES +
		((crd.diagonal >= w->pyraminx.size) ? UP : DOWN);
	*position = toPosition((PyraminxWidget) w, crd);
	return True;
}

void
narrowSelection2D(Pyraminx2DWidget w, int style,
		int *face, int *position, int *direction) {
	if (style == PERIOD2) {
		if (*direction == CW)
			*direction = TR;
		else if (*direction == CCW)
			*direction = BL;
	} else {	/* style == PERIOD3 */
		if (*direction == CW || *direction == CCW) {
			CRD crd;
			toCRD((PyraminxWidget) w, *face, *position, &crd);
			crd.diagonal = w->pyraminx.size -
				((crd.diagonal < w->pyraminx.size) ?
				UP : DOWN);
			*direction = ((*direction == CW &&
				crd.diagonal == w->pyraminx.size) ||
				(*direction == CCW &&
				crd.diagonal != w->pyraminx.size))
				? TR : BL;
			*face = !(*face % 2) + 2 * (*face / 2);
			crd.row = w->pyraminx.size - 1;
			crd.column = 0;
			*position = toPosition((PyraminxWidget) w, crd);
		}
	}
}

#ifndef WINVER
static Boolean
setValuesPuzzle2D(Widget current, Widget request, Widget renew)
{
	Pyraminx2DWidget c = (Pyraminx2DWidget) current, w = (Pyraminx2DWidget) renew;
	Boolean redraw = False;
	if (w->pyraminx.size != c->pyraminx.size ||
			w->pyraminx.sticky != c->pyraminx.sticky) {
		resetPieces((PyraminxWidget) w);
		resizePuzzle2D(w);
		redraw = True;
	}
	if (w->pyraminx.facetLength != c->pyraminx.facetLength) {
		resizePuzzle2D(w);
		redraw = True;
	}
	return (redraw);
}
#endif

#ifndef WINVER
static
#endif
void
resizePuzzle2D(Pyraminx2DWidget w)
{
	int tempLength;
#ifdef WINVER
	RECT rect;

	/* Determine size of client area */
	(void) GetClientRect(w->core.hWnd, &rect);
	w->core.width = rect.right;
	w->core.height = rect.bottom;
#endif

	w->pyraminx.delta = 4;
	w->pyraminx.vertical = (w->core.height >= w->core.width);
	if (w->pyraminx.vertical)
		tempLength = MIN(w->core.height / 2, w->core.width);
	else
		tempLength = MIN(w->core.height, w->core.width / 2);
	w->pyraminx.facetLength = MAX((tempLength - w->pyraminx.delta + 1) /
		w->pyraminx.size, 0);
	w->pyraminx.faceLength = w->pyraminx.size * w->pyraminx.facetLength;
	w->pyraminx.viewLength = w->pyraminx.faceLength + w->pyraminx.delta + 3;
	if (w->pyraminx.vertical) {
		w->pyraminx.puzzleSize.x = w->pyraminx.viewLength - 1;
		w->pyraminx.puzzleSize.y = 2 * w->pyraminx.viewLength -
			w->pyraminx.delta - 2;
	} else {
		w->pyraminx.puzzleSize.x = 2 * w->pyraminx.viewLength -
			w->pyraminx.delta - 2;
		w->pyraminx.puzzleSize.y = w->pyraminx.viewLength - 1;
	}
	w->pyraminx.puzzleOffset.x = ((int) w->core.width -
		w->pyraminx.puzzleSize.x) / 2;
	w->pyraminx.puzzleOffset.y = ((int) w->core.height -
		w->pyraminx.puzzleSize.y) / 2;
	resizePieces(w);
}

#ifndef WINVER
static
#endif
void
initializePuzzle2D(
#ifdef WINVER
Pyraminx2DWidget w
#else
Widget request, Widget renew
#endif
)
{
#ifndef WINVER
	Pyraminx2DWidget w = (Pyraminx2DWidget) renew;

	setAllColors((PyraminxWidget) w);
#endif
	w->pyraminx.dim = 2;
	resizePuzzle2D(w);
}

#ifndef WINVER
static
#endif
void
exposePuzzle2D(
#ifdef WINVER
Pyraminx2DWidget w
#else
Widget renew, XEvent *event, Region region
#endif
)
{
#ifndef WINVER
	Pyraminx2DWidget w = (Pyraminx2DWidget) renew;

	if (!w->core.visible)
		return;
#endif
	eraseFrame2D(w);
	drawFrame2D(w, w->pyraminx.focus);
	drawAllPieces((PyraminxWidget) w);
}

#ifndef WINVER
static void
movePuzzle2DTop(Pyraminx2DWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((PyraminxWidget) w, event->xbutton.x, event->xbutton.y, TOP,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask));
}

static void
movePuzzle2DTr(Pyraminx2DWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((PyraminxWidget) w, event->xbutton.x, event->xbutton.y, TR,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask));
}

static void
movePuzzle2DLeft(Pyraminx2DWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((PyraminxWidget) w, event->xbutton.x, event->xbutton.y, LEFT,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask));
}

static void
movePuzzle2DRight(Pyraminx2DWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((PyraminxWidget) w, event->xbutton.x, event->xbutton.y, RIGHT,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask));
}

static void
movePuzzle2DBl(Pyraminx2DWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((PyraminxWidget) w, event->xbutton.x, event->xbutton.y, BL,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask));
}

static void
movePuzzle2DBottom(Pyraminx2DWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((PyraminxWidget) w, event->xbutton.x, event->xbutton.y, BOTTOM,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask));
}
#endif
